/** @file   proctoralien.cpp
 * @brief   Implementation of ProctorAlien - class.
 * @version $Revision: 1.4 $
 * @author  Tomi Lamminsaari
 */

#include "proctoralien.h" // class's header file
#include "proctorai.h"
#include "soundsamples.h"
#include "animplayer.h"
#include "gameanims.h"
#include "GfxManager.h"
#include "www_map.h"
#include "settings.h"
#include "warglobals.h"
#include "AnimId.h"
using namespace eng2d;


namespace WeWantWar {


const int ProctorAlien::GUN_CONTROLPOINT_INDEX = 0;


/** Constructor.
 */
ProctorAlien::ProctorAlien() :
  Alien( ),
  
  m_speedVec( 0, 0 )
{
  ObjectID::Type oid = ObjectID::TYPE_PROCTORALIEN;
  
  m_speedVec.y( - Settings::floatObjProp(oid, "spd_walk:") );
  
  this->setCorrectAnimation( GameAnims::EIdle );
  
  this->boundingSphere( Settings::floatObjProp(oid, "bounding_sphere:") );
  this->setArmor( Settings::floatObjProp(oid, "armor:") );
  
  this->setCollisionPoint( 0, Vec2D( -13,-13 ) );
  this->setCollisionPoint( 1, Vec2D(  13,-13 ) );
  this->setCollisionPoint( 2, Vec2D(  13, 13 ) );
  this->setCollisionPoint( 3, Vec2D( -13, 13 ) );
  
  // Add the controlpoint for the shogun head
  this->addCtrlPoint( Vec2D( 2, -18 ) );
  
  // Create the ai-object.
  this->setController( new ProctorAI( this ) );
}



/** Destructor.
 */
ProctorAlien::~ProctorAlien()
{
}



/** THe update-method
 */
void ProctorAlien::update()
{
  if ( this->state() == GameObject::STATE_KILLED ) {
    return;
  }
  
  if ( this->state() == STATE_DYING ) {
    // We're dying, so update the dying animation
    if ( m_animation.paused() ) {
      // The dying animation has been run through, so set this object
      // to killed-state
      this->state( STATE_KILLED );
      this->setCorrectAnimation( GameAnims::EKilled );
    }
    return;

  }

  m_pController->update();

  this->changeAngle( m_pController->turn() );
  if ( m_pController->forward() == 1 ) {
    Vec2D spdV( m_speedVec );
    spdV.rotate( this->angle() );
    this->move( spdV );
  }

  if ( m_pController->shoot() == 1 ) {
    this->attack();

  } else {
    // We're not shooting
    if ( this->getAnimID() != GameAnims::EWalk && m_pController->forward() != 0 ) {
      this->setCorrectAnimation( GameAnims::EWalk );
    }
  }
}


/** Kills this alien
 */
void ProctorAlien::kill()
{
  this->state( GameObject::STATE_DYING );
  this->setCorrectAnimation( GameAnims::EDying );
  this->makeSound( GameObject::SND_DIE );
}



/** Makes sound
 */
void ProctorAlien::makeSound( GameObject::SoundID id ) const
{
  if ( id == GameObject::SND_ATTACK ) {
    Sound::playSample( SMP_WEAPON2, false );
  }
  if ( id == GameObject::SND_PAIN ) {
    int t = ( rand() % 2 < 1 ) ? SMP_ALIEN1_PAIN : SMP_ALIEN1_PAIN2;
    Sound::playSample( t, false );
  }
  if ( id == GameObject::SND_DIE ) {
    Sound::playSample( SMP_ALIEN1_DIE, false );
  }
}



/** Bullet hit
 */
bool ProctorAlien::hitByBullet( Bullet* pB )
{
  bool ret = Alien::hitByBullet( pB );
  if ( ret == true ) {
    if ( pB->iType != Bullet::EFlameThrower ) {
      ParticleBlood* pP = new ParticleBlood( pB->iPosition, pB->velocity(),
                                             12, Color(130,60,10) );
      WarGlobals::pPartManager->addSystem( pP );
    }
  }
  return ret;
}



/** Causes some damage
 */
bool ProctorAlien::causeDamage( Bullet* pB )
{
  // We inform the AI-controller.
  ProctorAI* pC = dynamic_cast<ProctorAI*>( m_pController );
  pC->setAlerted( true );
  
  // And let the parent class handle the damage.
  return Alien::causeDamage( pB );
}



/** Returns the type of this alien
 */
ObjectID::Type ProctorAlien::objectType() const
{
  return ObjectID::TYPE_PROCTORALIEN;
}



/** Sets the requested animation
 */
void ProctorAlien::setCorrectAnimation( int aAnimId )
{
  if ( this->getAnimID() != aAnimId ) {
    const Animation& anim = GameAnims::findAnimation( AnimId::KProctorAlien, aAnimId );
    this->setAnimation( anim, aAnimId );
  }
}



/** Shoots
 */
void ProctorAlien::attack()
{
  if ( this->reloading() == false ) {
    // Handle shooting

    Vec2D gunV = this->getCtrlPoint( GUN_CONTROLPOINT_INDEX );
    gunV += this->position();

    Weapon::Specs& w = Weapon::getWeaponSpecs( Weapon::W_SHOTGUN );
    for ( int i=0; i < 4; i++ ) {
      Bullet* b = BulletTable::createBullet( this, gunV, Bullet::EShotgun );
      b->iDamage /= 2;
      b->iRange *= 0.9;

      WarGlobals::pBulletManager->spawnBullet( b );
    }
    Vec2D dirV( 0, -16 );
    dirV.rotate( this->angle() );
    ParticleSystem* particles = new ParticleGunFlames( gunV, dirV, 15, 60 );
    WarGlobals::pPartManager->addSystem( particles );
    const Animation& flameAnim = GameAnims::findAnimation(AnimId::KShotgunShootFlame);
    AnimPlayer::spawn( flameAnim, gunV, 0 );
    const Animation& lightAnim = GameAnims::findAnimation( AnimId::KRifleShotLight );
    if ( Settings::shootingLights == true ) {
      AnimPlayer::spawn( lightAnim, gunV, 0 );
    }
    this->setCounter( RELOAD_COUNTER_INDEX, w.reload );
    this->makeSound( GameObject::SND_ATTACK );
    this->setCorrectAnimation( GameAnims::EShoot );
  }
}


} // end of namespace

